# -*- coding: utf-8 -*-
"""
/***************************************************************************
 GWSAlkisGeocoderDialog
                                 A QGIS plugin
 Adress Geocoding using GBD WebSuite and ALKIS database
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2019-06-26
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Geoinformatikbüro Dassau GmbH
        email                : info@gbd-consult.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import json
from PyQt5 import uic
from PyQt5.QtWidgets import QApplication, QDialog
from PyQt5.QtNetwork import QNetworkRequest
from PyQt5.QtCore import (
    QVariant,
    Qt,
    QCoreApplication,
    QByteArray,
    QUrl)
from qgis.utils import iface
from qgis.core import (
    QgsApplication,
    QgsNetworkAccessManager,
    QgsAuthMethodConfig,
    QgsField,
    QgsProject,
    QgsVectorLayer,
    QgsGeometry,
    QgsPointXY)

# This loads your .ui file so that PyQt can populate your plugin with the
# elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'alkis_geocoder_dialog_base.ui'))


class AlkisGeocoderDialog(QDialog, FORM_CLASS):
    """Dialog for the Alkis Geocoder Plugin."""
    def __init__(self, parent=None):
        """Constructor."""
        super(AlkisGeocoderDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        self.config = {}
        self.authcfg = None
        self.gws_url = None
        self.authCfgSelect.selectedConfigIdChanged.connect(self.update_auth)

        self.config_path = os.path.join(
                        QgsApplication.qgisSettingsDirPath(),
                        'GBD_WebSuite',
                        'gws.json')

        if os.path.exists((self.config_path)):
            with open(self.config_path, 'r') as fp:
                self.config = json.load(fp)
        elif not os.path.exists(os.path.dirname(self.config_path)):
            os.makedirs(os.path.dirname(self.config_path))

        if self.config.get('authcfg'):
            self.authCfgSelect.setConfigId(self.config.get('authcfg'))

        # Populate help window
        with open(os.path.join(
                os.path.dirname(__file__), 'help.html')) as html:
            self.textBrowser.setHtml(html.read())

        # Only show delimitedtext layers
        self.set_layer_exceptions()

        self.on_layer_change(self.tableLayer.currentLayer())
        self.tableLayer.layerChanged.connect(self.on_layer_change)

        self.generateLayerButton.clicked.connect(self.generate_layer)
        self.generateLayerButton.setEnabled(False)

        if not self.tableLayer.currentLayer():
            self.attributeBox.setEnabled(False)

        QgsProject.instance().layerWasAdded.connect(
            lambda x: self.set_layer_exceptions())
        QgsProject.instance().layerRemoved.connect(
            lambda x: self.set_layer_exceptions())

        self.hausnummerField.fieldChanged.connect(
            lambda x: self.check_input_fields())
        self.strasseField.fieldChanged.connect(
            lambda x: self.check_input_fields())
        self.gemarkungField.fieldChanged.connect(
            lambda x: self.check_input_fields())
        self.authCfgSelect.selectedConfigIdChanged.connect(
            lambda x: self.check_input_fields())

    def update_auth(self, authcfg):
        """update the auth config."""
        conf = QgsAuthMethodConfig()
        QgsApplication.authManager().loadAuthenticationConfig(
            authcfg, conf, True)
        if conf.uri():
            self.gws_url = QUrl(conf.uri())
            self.authcfg = authcfg
            self.config.update({
                'authcfg': authcfg
            })
            with open(self.config_path, 'w') as fp:
                json.dump(self.config, fp)
        else:
            self.gws_url = None
            self.authcfg = None

    def geocode_addresses(self, address_list):
        """Geocode a list of addresses using GWS."""
        req = QNetworkRequest(self.gws_url)
        req.setRawHeader(QByteArray(b"Content-Type"),
                         QByteArray(b"application/json"))
        nam = QgsNetworkAccessManager.instance()
        data = QByteArray(json.dumps({
            "cmd": "alkisgeocoderDecode",
            "params": {
                "crs": "EPSG:25832",
                "adressen": address_list
            }
        }).encode())
        return nam.blockingPost(req, data, self.authcfg)

    def set_layer_exceptions(self):
        """Show only delimeted text layers in the layer selector."""
        excepted = []
        for layer in QgsProject.instance().mapLayers().values():
            if hasattr(layer, 'providerType') and \
                    layer.providerType() not in ('delimitedtext', 'ogr'):
                excepted.append(layer)
            elif layer.geometryType() != 4:
                excepted.append(layer)
        self.tableLayer.setExceptedLayerList(excepted)

    def check_input_fields(self):
        """Check all required input fields of the form."""
        if self.tableLayer.currentLayer() \
                and self.strasseField.currentField() \
                and self.gemarkungField.currentField() \
                and self.authcfg \
                and self.gws_url \
                and self.hausnummerField.currentField():
            self.generateLayerButton.setEnabled(True)
        else:
            self.generateLayerButton.setEnabled(False)

    def on_layer_change(self, layer):
        """run when the active layer of the QgsMapLayerCombobox changes."""
        if self.tableLayer.currentLayer():
            self.generateLayerButton.setEnabled(True)
            self.attributeBox.setEnabled(True)

        self.strasseField.setLayer(layer)
        self.hausnummerField.setLayer(layer)
        self.gemarkungField.setLayer(layer)

        self.check_input_fields()

    def generate_layer(self):
        """ The main function, that does all the geocoding work. """

        # disable the Button while generating
        self.setEnabled(False)
        QApplication.setOverrideCursor(Qt.WaitCursor)
        QCoreApplication.processEvents()

        # create new memory layer
        layer = self.tableLayer.currentLayer()
        features = [f for f in layer.getFeatures()]
        mem_layer = QgsVectorLayer(
            'Point?crs=epsg:25832',
            '{}_geocoded'.format(layer.name()),
            'memory')
        mem_layer_data = mem_layer.dataProvider()
        attr = layer.dataProvider().fields().toList()
        mem_layer_data.addAttributes(
            attr + [
                QgsField('lat', QVariant.Double),
                QgsField('lon', QVariant.Double)
                ])
        mem_layer.updateFields()
        mem_layer_data.addFeatures(features)
        mem_layer.startEditing()

        addr_list = []
        fid_list = []
        for feature in mem_layer.getFeatures():
            gemarkung = feature[self.gemarkungField.currentField()]
            strasse = feature[self.strasseField.currentField()]
            hausnummer = feature[self.hausnummerField.currentField()]
            if gemarkung and strasse and hausnummer:
                addr_list.append({
                    'gemeinde': gemarkung,
                    'strasse': strasse,
                    'hausnummer': str(hausnummer)

                })
                fid_list.append(feature.id())

        # CHUNKSIZE
        n= 100
        addr_chunks = [addr_list[i:i+n] for i in range(0, len(addr_list), n)]
        fid_chunks = [fid_list[i:i+n] for i in range(0, len(fid_list), n)]

        result = []
        for addr_chunk, fid_chunk in zip(addr_chunks, fid_chunks):
            reply = self.geocode_addresses(addr_chunk)
            if reply.error() != 0:
                if reply.error() == 3:
                    # Host not found
                    iface.messageBar().pushCritical(
                        'Netzwerkfehler',
                        'Host {} nicht gefunden'.format(str(self.gws_url)))
                    break
                elif reply.error() == 201:
                    # Access denied (403)
                    iface.messageBar().pushCritical(
                        'GWS Fehler!', 'falsche Zugangsdaten')
                    break
                else:
                    # Some other error
                    iface.messageBar().pushCritical(
                        'Netzwerkfehler',
                        'Code {}'.format(reply.error()))
                    break
            else: # No Error
                coordinates = json.loads(
                    str(reply.content(), 'utf-8')).get('coordinates')
                if not coordinates:
                    iface.messageBar().pushCritical(
                        'GWS Fehler!',
                        'GWS unterstützt kein AlkisGeocoder!')
                    break
                else:
                    for item in zip(fid_chunk, coordinates):
                        result.append(item)
            
        for (fid, coords) in result:
            if coords:
                feature = mem_layer.getFeature(fid)
                x = coords[0]
                y = coords[1]
                feature.setAttribute('lat', x)
                feature.setAttribute('lon', y)
                geom = QgsPointXY(x, y)
                feature.setGeometry(QgsGeometry.fromPointXY(geom))
                mem_layer.updateFeature(feature)

        geom_features = len(list(filter(
            lambda x: x.hasGeometry(),
            [f for f in mem_layer.getFeatures()])))
        if geom_features > 0:
            iface.messageBar().pushSuccess(
                'Geocodierung abgeschlossen',
                '%s von %s Features konnten geocodiert werden'
                % (geom_features, layer.featureCount()))
            mem_layer.commitChanges()
            QgsProject.instance().addMapLayer(mem_layer)
        else:
            del mem_layer
            iface.messageBar().pushCritical(
                'Geocodierung fehlgeschlagen',
                'Es konnten keine Features geocodiert werden.')

        self.setEnabled(True)
        QApplication.setOverrideCursor(Qt.ArrowCursor)
